<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Apishowroom;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ShowroomManagementController extends Controller
{
    public function index()
    {
        $apishowrooms = Apishowroom::paginate(20);
        return view('admin.sync.dashboard', compact('apishowrooms'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'identifier' => 'required|string|max:255|unique:apishowrooms,identifier',
            'api_url' => 'required|url|max:255',
            'api_token' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        Apishowroom::create([
            'name' => $validated['name'],
            'identifier' => $validated['identifier'],
            'api_url' => $validated['api_url'],
            'api_token' => $validated['api_token'],
            'is_active' => $validated['status'] === 'active',
            'last_pulled_at' => null,
        ]);

        return redirect()->route('admin.sync.dashboard')->with('success', 'Showroom added successfully!');
    }

    public function update(Request $request, Apishowroom $showroom)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'identifier' => 'required|string|max:255|unique:apishowrooms,identifier,' . $showroom->id,
            'api_url' => 'required|url|max:255',
            'api_token' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        $showroom->update([
            'name' => $validated['name'],
            'identifier' => $validated['identifier'],
            'api_url' => $validated['api_url'],
            'api_token' => $validated['api_token'],
            'is_active' => $validated['status'] === 'active',
        ]);

        return redirect()->route('admin.sync.dashboard')->with('success', 'Showroom updated successfully!');
    }

    public function destroy(Apishowroom $showroom)
    {
        $showroom->delete();
        return redirect()->route('admin.sync.dashboard')->with('success', 'Showroom deleted successfully!');
    }

    public function toggleStatus(Apishowroom $showroom)
    {
        $showroom->is_active = !$showroom->is_active;
        $showroom->save();

        return redirect()->route('admin.sync.dashboard')->with('success', 'Showroom status updated!');
    }

public function pullSingleStock(Request $request, Apishowroom $showroom)
{
    $results = [];

    $from = $request->input('from', now()->subDay()->toDateString());
    $to   = $request->input('to', now()->toDateString());

    try {
        $response = Http::timeout(30)
            ->withToken($showroom->api_token)
            ->get("{$showroom->api_url}/data/stock", [
                'from' => $from,
                'to'   => $to,
            ]);

        if ($response->successful()) {
            $payload = $response->json();
            $stockItems = $payload['items'] ?? [];
            Log::debug("Received stock data from {$showroom->name}:", $stockItems);

            foreach ($stockItems as $item) {
                \App\Models\ProductStock::updateOrCreate(
                    [
                        'product_id'  => $item['product_id'],
                        'showroom_id' => $showroom->id,
                        'stock_date'  => \Carbon\Carbon::parse($item['stock_date'])->startOfDay(),
                    ],
                    [
                        'opening_stock' => $item['opening_stock'] ?? 0,
                        'received'      => $item['received'] ?? 0,
                        'return_damage' => $item['return_damage'] ?? 0,
                        'closing_stock' => $item['closing_stock'] ?? 0,
                    ]
                );
            }

            $showroom->update(['last_pulled_at' => now()]);
            Log::info("Pulled and saved stock from {$showroom->name}", ['items_synced' => count($stockItems)]);

            $results[] = ['showroom' => $showroom->name, 'status' => 'Success', 'items' => count($stockItems)];
        } else {
            $results[] = ['showroom' => $showroom->name, 'status' => 'Failed', 'error' => "HTTP {$response->status()}"];
            Log::error("Failed to pull stock from {$showroom->name}", ['status' => $response->status()]);
        }
    } catch (\Exception $e) {
        $results[] = ['showroom' => $showroom->name, 'status' => 'Error', 'error' => $e->getMessage()];
        Log::error("Exception during stock pull from {$showroom->name}", ['error' => $e->getMessage()]);
    }

    return redirect()->route('admin.sync.dashboard')
        ->with('sync_results', $results)
        ->with('success', 'Stock sync completed.');
}


public function pullAllStock(Request $request)
{
    $results = [];
    $from = $request->input('from', now()->subDay()->toDateString());
    $to   = $request->input('to', now()->toDateString());

    $showrooms = Apishowroom::where('is_active', true)->get();

    foreach ($showrooms as $showroom) {
        try {
            $response = Http::timeout(30)
                ->withToken($showroom->api_token)
                ->get("{$showroom->api_url}/data/stock", [
                    'from' => $from,
                    'to'   => $to,
                ]);

            if ($response->successful()) {
                $payload = $response->json();
                $stockItems = $payload['items'] ?? [];

                foreach ($stockItems as $item) {
                    \App\Models\ProductStock::updateOrCreate(
                        [
                            'product_id'  => $item['product_id'],
                            'showroom_id' => $showroom->id,
                            'stock_date'  => \Carbon\Carbon::parse($item['stock_date'])->startOfDay(),
                        ],
                        [
                            'opening_stock' => $item['opening_stock'] ?? 0,
                            'received'      => $item['received'] ?? 0,
                            'return_damage' => $item['return_damage'] ?? 0,
                            'closing_stock' => $item['closing_stock'] ?? 0,
                        ]
                    );
                }

                $showroom->update(['last_pulled_at' => now()]);
                Log::info("Pulled and saved stock from {$showroom->name}", ['items_synced' => count($stockItems)]);

                $results[] = ['showroom' => $showroom->name, 'status' => 'Success', 'items' => count($stockItems)];
            } else {
                $results[] = ['showroom' => $showroom->name, 'status' => 'Failed', 'error' => "HTTP {$response->status()}"];
                Log::error("Failed to pull stock from {$showroom->name}", ['status' => $response->status()]);
            }
        } catch (\Exception $e) {
            $results[] = ['showroom' => $showroom->name, 'status' => 'Error', 'error' => $e->getMessage()];
            Log::error("Exception pulling from {$showroom->name}", ['error' => $e->getMessage()]);
        }
    }

    return redirect()->route('admin.sync.dashboard')
        ->with('sync_results', $results)
        ->with('success', 'Stock sync completed.');
}



}
