<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ProductStock;
use App\Models\Showroom;
use App\Models\Product;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class CentralStockReportController extends Controller
{
    // Central office user must be authenticated and authorized to access these endpoints
    // You might add additional permission checks here using Laravel Gates/Policies

    public function getShowrooms()
    {
        $showrooms = Showroom::select('id', 'name')->orderBy('name')->get();
        return response()->json($showrooms);
    }

    public function getDetailedStockReport(Request $request)
    {
        $request->validate([
            'date_from' => 'nullable|date_format:Y-m-d',
            'date_to' => 'nullable|date_format:Y-m-d|after_or_equal:date_from',
            'showroom_ids' => 'nullable|array',
            'showroom_ids.*' => 'integer|exists:showrooms,id',
            'product_ids' => 'nullable|array',
            'product_ids.*' => 'integer|exists:products,id',
            'product_category_id' => 'nullable|integer|exists:product_categories,id', // Assuming you have a Product Category model/table
            'per_page' => 'nullable|integer|min:1|max:100',
            'page' => 'nullable|integer|min:1',
        ]);

        $query = ProductStock::with(['product', 'showroom']);

        // Apply date filters
        if ($request->has('date_from')) {
            $query->whereDate('stock_date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('stock_date', '<=', $request->date_to);
        } else {
            // If only date_from is provided, assume it's for that specific date
            // Or default to today if no date is given. Choose your default logic.
            if (!$request->has('date_from')) {
                $query->whereDate('stock_date', Carbon::today());
            }
        }

        // Apply showroom filter
        if ($request->has('showroom_ids')) {
            $query->whereIn('showroom_id', $request->showroom_ids);
        }

        // Apply product filter
        if ($request->has('product_ids')) {
            $query->whereIn('product_id', $request->product_ids);
        }

        // Apply product category filter (assuming product has product_category_id)
        if ($request->has('product_category_id')) {
            $query->whereHas('product', function ($q) use ($request) {
                $q->where('product_category_id', $request->product_category_id);
            });
        }

        // Order results (important for consistency)
        $query->orderBy('stock_date', 'desc')
              ->orderBy('showroom_id')
              ->orderBy('product_id');

        // Paginate results for large datasets
        $perPage = $request->input('per_page', 25);
        $stocks = $query->paginate($perPage);

        // Transform data for presentation (optional, but good practice for APIs)
        $formattedStocks = $stocks->through(function ($stock) {
            $saleQty = $stock->opening_stock + $stock->received - $stock->return_damage - $stock->closing_stock;
            return [
                'id' => $stock->id,
                'stock_date' => $stock->stock_date->format('Y-m-d'),
                'showroom' => [
                    'id' => $stock->showroom->id,
                    'name' => $stock->showroom->name,
                ],
                'product' => [
                    'id' => $stock->product->id,
                    'name' => $stock->product->productname,
                    'price' => $stock->product->price,
                    'category' => $stock->product->productcategory, // Or eager load product->category_relation->name
                ],
                'opening_stock' => $stock->opening_stock,
                'received' => $stock->received,
                'return_damage' => $stock->return_damage,
                'closing_stock' => $stock->closing_stock,
                'sale_quantity' => $saleQty,
                'opening_value' => $stock->opening_stock * $stock->product->price,
                'received_value' => $stock->received * $stock->product->price,
                'return_damage_value' => $stock->return_damage * $stock->product->price,
                'closing_value' => $stock->closing_stock * $stock->product->price,
                'sale_value' => $saleQty * $stock->product->price,
            ];
        });

        return response()->json($formattedStocks);
    }

    public function getSummaryReport(Request $request)
    {
        $request->validate([
            'date_from' => 'required|date_format:Y-m-d', // Summary usually needs a date range
            'date_to' => 'required|date_format:Y-m-d|after_or_equal:date_from',
            'showroom_ids' => 'nullable|array',
            'showroom_ids.*' => 'integer|exists:showrooms,id',
        ]);

        $dateFrom = Carbon::parse($request->date_from);
        $dateTo = Carbon::parse($request->date_to);

        $query = ProductStock::whereBetween('stock_date', [$dateFrom, $dateTo])
            ->join('products', 'product_stocks.product_id', '=', 'products.id');

        if ($request->has('showroom_ids')) {
            $query->whereIn('showroom_id', $request->showroom_ids);
        }

        // Group by showroom and aggregate totals for the period
        $summary = $query->select(
            'showroom_id',
            DB::raw('SUM(opening_stock * products.price) as total_opening_value'),
            DB::raw('SUM(received * products.price) as total_received_value'),
            DB::raw('SUM(return_damage * products.price) as total_return_damage_value'),
            DB::raw('SUM(closing_stock * products.price) as total_closing_value'),
            DB::raw('SUM((opening_stock + received - return_damage - closing_stock) * products.price) as total_sale_value')
        )
        ->groupBy('showroom_id')
        ->get();

        // Append showroom names
        $summary->load('showroom'); // Eager load showroom relation

        return response()->json($summary->map(function($item) {
            return [
                'showroom_id' => $item->showroom_id,
                'showroom_name' => $item->showroom->name,
                'total_opening_value' => round($item->total_opening_value, 2),
                'total_received_value' => round($item->total_received_value, 2),
                'total_return_damage_value' => round($item->total_return_damage_value, 2),
                'total_closing_value' => round($item->total_closing_value, 2),
                'total_sale_value' => round($item->total_sale_value, 2),
            ];
        }));
    }
}