<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\User;
use App\Models\Apishowroom; // <--- Changed from App\Models\Showroom
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ProductPriceApiController extends Controller
{
    public function syncPricesToShowroom(Request $request)
    {
        // 1. Validate the incoming request data
        //    Ensures that at least one valid showroom ID is selected.
        $request->validate([
            'apishowroom_ids' => ['required', 'array', 'min:1'],
            'apishowroom_ids.*' => ['exists:apishowrooms,id'], // Ensures IDs exist in your apishowrooms table
        ]);

        $targetApishowroomIds = $request->input('apishowroom_ids');

        // 2. Fetch the product price data to be sent
        //    We only send active prices (null end date) or future prices (end date in future).
        //    Historical prices are typically not pushed for active management.
        $productPrices = ProductPrice::where(function ($query) {
                                    $query->whereNull('effective_end_date')
                                          ->orWhere('effective_end_date', '>', now()); // 'now()' is current time
                                })
                                ->orderBy('product_id')
                                ->orderBy('effective_start_date')
                                ->get()
                                ->map(function ($price) {
                                    // Transform the Eloquent model into a plain array
                                    // with specific fields, ensuring correct data types for JSON.
                                    return [
                                        'id' => $price->id,
                                        'product_id' => $price->product_id,
                                        'price' => (float)$price->price, // Cast to float for consistency
                                        'effective_start_date' => $price->effective_start_date->toDateTimeString(),
                                        'effective_end_date' => $price->effective_end_date ? $price->effective_end_date->toDateTimeString() : null,
                                        'changed_by_user_id' => $price->changed_by_user_id,
                                        'reason' => $price->reason,
                                        'is_showroom_override' => (bool)$price->is_showroom_override, // Cast to boolean
                                        'showroom_id' => $price->showroom_id, // This is the ID from your general 'showrooms' table
                                    ];
                                })->toArray(); // Convert the collection of arrays to a pure PHP array

        // 3. Fetch the API showroom details based on selected IDs
        //    This uses your dedicated `Apishowroom` model to get API URLs and tokens.
        $apishowroomsToSync = Apishowroom::whereIn('id', $targetApishowroomIds)->get();
        $syncResults = []; // To store messages for the user about each sync operation

        // 4. Iterate through each selected showroom and send the data
        foreach ($apishowroomsToSync as $apishowroom) {
            // Ensure the showroom has an API URL configured
            if ($apishowroom->api_url) {
                try {
                    // Initialize Laravel's HTTP Client
                    // Set a timeout, especially for larger data sets or slower networks
                    $http = Http::timeout(60); // 60 seconds timeout

                    // Attach API token if available in the Apishowroom model
                    // This assumes the showroom's API expects a Bearer token.
                    if ($apishowroom->api_token) {
                        $http->withToken($apishowroom->api_token);
                        Log::debug("Central: Sending API token for showroom {$apishowroom->name}.");
                    }

                    // Make the POST request to the showroom's API endpoint
                    $response = $http->post($apishowroom->api_url, [
                        'prices' => $productPrices, // Send the prepared price data
                    ]);

                    // Check the response from the showroom's API
                    if ($response->successful()) {
                        $syncResults[] = "Successfully pushed prices to showroom: {$apishowroom->name}";
                        Log::info("Central: Pushed prices to showroom {$apishowroom->name} (URL: {$apishowroom->api_url}). Response: " . $response->body());

                        // Optional: Update the last_pulled_at timestamp in central for this Apishowroom
                        // This helps track the last successful sync time.
                        $apishowroom->update(['last_pulled_at' => now()]);

                    } else {
                        // Log and report errors from the showroom's API
                        $syncResults[] = "Failed to push prices to showroom: {$apishowroom->name}. Status: {$response->status()}, Error: " . $response->body();
                        Log::error("Central: Failed to push prices to showroom {$apishowroom->name} (URL: {$apishowroom->api_url}). Status: {$response->status()}, Response: " . $response->body());
                    }
                } catch (\Exception $e) {
                    // Catch network errors, connection timeouts, or other exceptions during the HTTP request
                    $syncResults[] = "Error pushing prices to showroom: {$apishowroom->name}. " . $e->getMessage();
                    Log::error("Central: Exception pushing prices to showroom {$apishowroom->name} (URL: {$apishowroom->api_url}): " . $e->getMessage(), ['exception' => $e]);
                }
            } else {
                // Handle cases where a showroom doesn't have an API URL configured
                $syncResults[] = "API Showroom '{$apishowroom->name}' does not have an API URL configured. Skipping.";
            }
        }

        // 5. Redirect back to the previous page (Product Prices index page)
        //    and flash the sync results to the session for display.
        //    This is the Post-Redirect-Get (PRG) pattern.
        return redirect()->back()->with('sync_results', $syncResults);
    }
    
}