<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Product;
use App\Models\Apishowroom; // Ensure this is the correct model for API-enabled showrooms
use Carbon\Carbon; // Ensure Carbon is imported if not already

class ProductSyncController extends Controller
{
    public function syncProductsToShowroom(Request $request)
    {
        $request->validate([
            'apishowroom_ids' => ['required', 'array', 'min:1'],
            'apishowroom_ids.*' => ['exists:apishowrooms,id'],
        ]);

        $targetApishowroomIds = $request->input('apishowroom_ids');

        $products = Product::orderBy('id')->get()->map(function ($product) {
            return [
                'id' => $product->id,
                'productname' => $product->productname,
                'productiondepartment' => $product->productiondepartment,
                'productcategory' => $product->productcategory,
                'price' => (float) $product->price,
                'dateofentry' => $product->dateofentry
                    ? Carbon::parse($product->dateofentry)->toDateTimeString()
                    : null,
                // New fields added here
                'barcode' => $product->barcode,
                'sku' => $product->sku,
                'custom_no' => $product->custom_no,
                'unit' => $product->unit,
                'status' => $product->status,
                'showroom_id' => $product->showroom_id, // This is the ID of the showroom this product belongs to internally
            ];
        })->toArray();

        $apishowroomsToSync = Apishowroom::whereIn('id', $targetApishowroomIds)->get();
        $syncResults = [];

        foreach ($apishowroomsToSync as $apishowroom) {
            if ($apishowroom->api_url) {
                try {

                    // We are overriding the URL from the database to point to the correct endpoint
                    // The rtrim function removes any trailing slashes to prevent double slashes in the URL
                    $correctUrl = rtrim($apishowroom->api_url, '/') . '/sync/products';


                    $http = Http::timeout(60);
                    if ($apishowroom->api_token) {
                        $http->withToken($apishowroom->api_token);
                    }

                    // Make the POST request to the corrected URL
                    $response = $http->post($correctUrl, [
                        'products' => $products,
                    ]);

                    // $response = $http->post($apishowroom->api_url, [
                    //     'products' => $products,
                    // ]);
                    Log::debug('Pushed products to ' . $apishowroom->name, ['payload' => $products, 'response_status' => $response->status(), 'response_body' => $response->body()]); // Enhanced logging

                    if ($response->successful()) {
                        $syncResults[] = "Synced products to showroom: {$apishowroom->name}";
                        Log::info("Product sync to {$apishowroom->name} succeeded.");
                        $apishowroom->update(['last_pulled_at' => now()]);
                    } else {
                        $syncResults[] = "Failed to sync to showroom: {$apishowroom->name}. Status: {$response->status()} Body: " . $response->body(); // Include response body for more context
                        Log::error("Product sync failed to {$apishowroom->name}: " . $response->body(), ['showroom_id' => $apishowroom->id, 'status' => $response->status()]);
                    }
                } catch (\Exception $e) {
                    $syncResults[] = "❌ Exception syncing to {$apishowroom->name}: " . $e->getMessage();
                    Log::error("Exception syncing to {$apishowroom->name}", ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString(), 'showroom_id' => $apishowroom->id]); // More detailed exception logging
                }
            } else {
                $syncResults[] = "Skipped showroom '{$apishowroom->name}': missing API URL.";
                Log::warning("Showroom '{$apishowroom->name}' skipped due to missing API URL.", ['showroom_id' => $apishowroom->id]);
            }
        }

        return response()->json(['sync_results' => $syncResults]);
    }
}
