<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ProductStock;
use App\Models\User; // Assuming your API tokens are tied to users, and users have a showroom_id
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ShowroomStockSubmissionController extends Controller
{
    public function submitDailyStock(Request $request)
    {
        // 1. Authenticate and identify the showroom
        $user = $request->user(); // This gets the authenticated user via Sanctum token

        // Ensure the authenticated user is linked to a showroom and authorized
        // You might have a specific role check here, e.g., if ($user->hasRole('showroom_reporter'))
        if (!$user || !$user->showroom_id) {
            return response()->json(['message' => 'Unauthorized or showroom not assigned to API user.'], 403);
        }

        $showroom_id = $user->showroom_id; // Get the showroom_id from the authenticated user

        // 2. Validate incoming data
        $data = $request->validate([
            'stock_date' => 'required|date_format:Y-m-d',
            'entries' => 'required|array',
            'entries.*.product_id' => 'required|integer|exists:products,id',
            'entries.*.opening_stock' => 'required|integer|min:0',
            'entries.*.received' => 'required|integer|min:0',
            'entries.*.return_damage' => 'required|integer|min:0',
            'entries.*.closing_stock' => 'required|integer|min:0',
            // Note: 'sale' is typically calculated, not submitted directly
        ]);

        // 3. Process data (using a database transaction for atomicity)
        DB::beginTransaction();
        try {
            foreach ($data['entries'] as $entry) {
                ProductStock::updateOrCreate(
                    [
                        'product_id' => $entry['product_id'],
                        'stock_date' => $data['stock_date'],
                        'showroom_id' => $showroom_id, // Crucially, use the authenticated showroom_id
                    ],
                    [
                        'opening_stock' => $entry['opening_stock'],
                        'received' => $entry['received'],
                        'return_damage' => $entry['return_damage'],
                        'closing_stock' => $entry['closing_stock']
                    ]
                );
            }
            DB::commit();

            Log::info("Stock submitted successfully for Showroom ID: {$showroom_id} on {$data['stock_date']}");
            return response()->json(['message' => 'Stock data submitted successfully.']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error submitting stock for Showroom ID: {$showroom_id} on {$data['stock_date']}: " . $e->getMessage());
            return response()->json(['message' => 'Failed to submit stock data.', 'error' => $e->getMessage()], 500);
        }
    }
}