<?php

namespace App\Http\Controllers;

use App\Models\ApiShowroom;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ApiDmySalesController extends Controller
{
    public function index(Request $request)
    {
        $showrooms = ApiShowroom::select('id', 'identifier', 'name')->get()->map(function ($showroom) {
            return ['id' => $showroom->identifier, 'name' => $showroom->name];
        })->toArray();

        $showroomId = $request->input('showroom_id', 'all');
        $apiShowroom = $showroomId === 'all' ? ApiShowroom::first() : ApiShowroom::where('identifier', $showroomId)->first();
        if (!$apiShowroom || !$apiShowroom->api_token) {
            return back()->with('error', 'No API token found for the selected Showroom.');
        }
        $baseUrl = rtrim($apiShowroom->api_url, '/');
        $apiPrefix = str_ends_with($apiShowroom->api_url, '/api') ? '' : '/api';
        $apiUrl = $baseUrl . $apiPrefix . '/dmy-sales';
        $apiToken = $apiShowroom->api_token;

        $queryParams = $request->only([
            'day', 'month', 'year', 'from_date', 'to_date',
            'productname', 'productcategory', 'productiondepartment',
            'barcode', 'sku', 'custom_no', 'showroom_id'
        ]);

        Log::info('API DMY Sales Request', [
            'url' => $apiUrl,
            'params' => $queryParams,
            'token' => substr($apiToken, 0, 10) . '...'
        ]);

        $response = Http::withToken($apiToken)->get($apiUrl, $queryParams);

        Log::info('Showroom API Response', [
            'status' => $response->status(),
            'body' => substr($response->body(), 0, 500),
            'headers' => $response->headers()
        ]);

        // Initialize filter variables to avoid undefined errors
        $productCategories = [];
        $productionDepartments = [];
        $productNames = [];

        if ($response->successful()) {
            $data = $response->json();
            if (!is_array($data) || !isset($data['report1'], $data['report2'], $data['filters_applied'])) {
                Log::error('Invalid Showroom API response structure', [
                    'data' => $data,
                    'response' => substr($response->body(), 0, 500)
                ]);
                return back()->with('error', 'Invalid or incomplete response from Showroom API.');
            }

            $report1 = $data['report1'];
            $report2 = $data['report2'];
            $filters = $data['filters_applied'];

            if (empty($report1['data']) && empty($report2['data'])) {
                return view('pos.index', compact(
                    'report1', 'report2', 'filters', 'showrooms',
                    'productCategories', 'productionDepartments', 'productNames'
                ))->with('warning', 'No data found for the selected filters.');
            }

            $productCategoriesResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/product-categories');
            $productionDepartmentsResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/production-departments');
            $productNamesResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/product-names');

            Log::info('Showroom Filter API Responses', [
                'categories' => $productCategoriesResponse->status(),
                'departments' => $productionDepartmentsResponse->status(),
                'names' => $productNamesResponse->status()
            ]);

            $productCategories = $productCategoriesResponse->successful() ? $productCategoriesResponse->json() : [];
            $productionDepartments = $productionDepartmentsResponse->successful() ? $productionDepartmentsResponse->json() : [];
            $productNames = $productNamesResponse->successful() ? $productNamesResponse->json() : [];

            return view('pos.index', compact(
                'report1', 'report2', 'filters', 'showrooms',
                'productCategories', 'productionDepartments', 'productNames'
            ));
        }

        Log::error('Showroom API request failed', [
            'status' => $response->status(),
            'body' => substr($response->body(), 0, 500)
        ]);

        // Return view with empty data to avoid breaking the UI
        $report1 = ['data' => [], 'columns' => [], 'total' => [], 'heading1' => '', 'heading2' => ''];
        $report2 = ['data' => [], 'columns' => [], 'total' => [], 'heading1' => '', 'heading2' => ''];
        $filters = $queryParams;

        return view('pos.index', compact(
            'report1', 'report2', 'filters', 'showrooms',
            'productCategories', 'productionDepartments', 'productNames'
        ))->with('error', 'Failed to fetch report from Showroom API: ' . $response->status());
    }

    public function create()
    {
        $showrooms = ApiShowroom::select('id', 'identifier', 'name')->get()->map(function ($showroom) {
            return ['id' => $showroom->identifier, 'name' => $showroom->name];
        })->toArray();

        $apiShowroom = ApiShowroom::first();
        if (!$apiShowroom || !$apiShowroom->api_token) {
            return back()->with('error', 'No API token found for Showroom.');
        }
        $apiToken = $apiShowroom->api_token;
        $baseUrl = rtrim($apiShowroom->api_url, '/');
        $apiPrefix = str_ends_with($apiShowroom->api_url, '/api') ? '' : '/api';

        $productCategoriesResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/product-categories');
        $productionDepartmentsResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/production-departments');
        $productNamesResponse = Http::withToken($apiToken)->get($baseUrl . $apiPrefix . '/product-names');

        Log::info('Showroom Filter API Responses (Create)', [
            'categories' => $productCategoriesResponse->status(),
            'departments' => $productionDepartmentsResponse->status(),
            'names' => $productNamesResponse->status()
        ]);

        $productCategories = $productCategoriesResponse->successful() ? $productCategoriesResponse->json() : [];
        $productionDepartments = $productionDepartmentsResponse->successful() ? $productionDepartmentsResponse->json() : [];
        $productNames = $productNamesResponse->successful() ? $productNamesResponse->json() : [];

        return view('pos.create', compact(
            'showrooms', 'productCategories', 'productionDepartments', 'productNames'
        ));
    }

    public function store(Request $request)
    {
        return redirect()->route('pos.index', $request->only([
            'day', 'month', 'year', 'from_date', 'to_date',
            'productname', 'productcategory', 'productiondepartment',
            'barcode', 'sku', 'custom_no', 'showroom_id'
        ]));
    }

    public function edit($id)
    {
        return redirect()->route('pos.index');
    }

    public function update(Request $request, $id)
    {
        return redirect()->route('pos.index');
    }

    public function destroy($id)
    {
        return redirect()->route('pos.index');
    }
}