<?php

namespace App\Http\Controllers;

use App\Models\Apishowroom;
use App\Models\Discount;
use App\Models\DiscountCampaign;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log; // For logging sync results

class DiscountCampaignApiController extends Controller
{
    /**
     * Trigger the synchronization of Discounts and Discount Campaigns to showrooms.
     * This method can be called via a web route.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function syncDiscountData()
    {
        $showrooms = Apishowroom::where('is_active', true)->get();
        $syncResults = [];

        if ($showrooms->isEmpty()) {
            return redirect()->back()->with('error', 'No active showrooms configured for synchronization.');
        }

        foreach ($showrooms as $showroom) {
            $showroomResults = [
                'name' => $showroom->name,
                'url' => $showroom->api_url,
                'discounts' => [],
                'campaigns' => [],
                'status' => 'success'
            ];

            Log::info("Starting sync to showroom: {$showroom->name} ({$showroom->api_url})");

            // Validate and normalize the API URL
            $baseUrl = rtrim($showroom->api_url, '/');

            // Ensure the base URL does not end with '/api' before appending it
            // This prevents http://localhost:8001/api/api/discounts
            if (substr($baseUrl, -4) === '/api') {
                $baseUrl = rtrim($baseUrl, '/api');
            }

            if (!filter_var($baseUrl, FILTER_VALIDATE_URL)) {
                $errorMessage = "Invalid API URL for showroom '{$showroom->name}': {$showroom->api_url}";
                $showroomResults['status'] = 'failure';
                $syncResults[] = $showroomResults;
                Log::error($errorMessage);
                continue; // Skip to the next showroom
            }

            // --- Sync Discounts ---
            $discounts = Discount::all();
            foreach ($discounts as $discount) {
                $endpoint = "{$baseUrl}/api/discounts"; // Correctly append /api/discounts
                try {
                    Log::info("  - Attempting to POST discount '{$discount->name}' to: {$endpoint}");
                    $response = Http::withToken($showroom->api_token)
                                    ->post($endpoint, $discount->toArray());

                    if ($response->successful()) {
                        $showroomResults['discounts'][] = ['name' => $discount->name, 'status' => 'synced'];
                        Log::info("  - Discount '{$discount->name}' synced to {$showroom->name}.");
                    } else {
                        $errorMessage = "Failed: " . ($response->json()['message'] ?? $response->body());
                        $showroomResults['discounts'][] = ['name' => $discount->name, 'status' => $errorMessage];
                        $showroomResults['status'] = 'partial_failure';
                        Log::error("  - Failed to sync discount '{$discount->name}' to {$showroom->name} (URL: {$endpoint}): " . $errorMessage);
                    }
                } catch (\Exception $e) {
                    $errorMessage = "Error: " . $e->getMessage();
                    $showroomResults['discounts'][] = ['name' => $discount->name, 'status' => $errorMessage];
                    $showroomResults['status'] = 'partial_failure';
                    Log::error("  - Exception syncing discount '{$discount->name}' to {$showroom->name} (URL: {$endpoint}): " . $errorMessage);
                }
            }

            // --- Sync Discount Campaigns ---
            $campaigns = DiscountCampaign::all();
            foreach ($campaigns as $campaign) {
                $endpoint = "{$baseUrl}/api/discount_campaigns"; // Correctly append /api/discount_campaigns
                try {
                    // Ensure product_ids is an array when sending, as it's cast to array in the model
                    $campaignData = $campaign->toArray();
                    // If product_ids is null, ensure it's an empty array for JSON encoding consistency
                    if (!isset($campaignData['product_ids'])) {
                        $campaignData['product_ids'] = [];
                    }

                    Log::info("  - Attempting to POST campaign '{$campaign->name}' to: {$endpoint}");
                    $response = Http::withToken($showroom->api_token)
                                    ->post($endpoint, $campaignData);

                    if ($response->successful()) {
                        $showroomResults['campaigns'][] = ['name' => $campaign->name, 'status' => 'synced'];
                        Log::info("  - Campaign '{$campaign->name}' synced to {$showroom->name}.");
                    } else {
                        $errorMessage = "Failed: " . ($response->json()['message'] ?? $response->body());
                        $showroomResults['campaigns'][] = ['name' => $campaign->name, 'status' => $errorMessage];
                        $showroomResults['status'] = 'partial_failure';
                        Log::error("  - Failed to sync campaign '{$campaign->name}' to {$showroom->name} (URL: {$endpoint}): " . $errorMessage);
                    }
                } catch (\Exception $e) {
                    $errorMessage = "Error: " . $e->getMessage();
                    $showroomResults['campaigns'][] = ['name' => $campaign->name, 'status' => $errorMessage];
                    $showroomResults['status'] = 'partial_failure';
                    Log::error("  - Exception syncing campaign '{$campaign->name}' to {$showroom->name} (URL: {$endpoint}): " . $errorMessage);
                }
            }

            // After syncing, update last_pulled_at for the showroom in the Central app
            $showroom->update(['last_pulled_at' => now()]);
            Log::info("Finished sync for showroom: {$showroom->name}.");
            $syncResults[] = $showroomResults;
        }

        // Prepare a summary message for the user
        $successCount = 0;
        $failureCount = 0;
        foreach ($syncResults as $result) {
            if ($result['status'] === 'success') {
                $successCount++;
            } else {
                $failureCount++;
            }
        }

        $message = "Synchronization complete. Successfully synced to {$successCount} showroom(s). ";
        if ($failureCount > 0) {
            $message .= "Failed or partially failed for {$failureCount} showroom(s). Check logs for details.";
            return redirect()->back()->with('warning', $message);
        }

        return redirect()->back()->with('success', $message);
    }
}
