<?php

namespace App\Http\Controllers;

use App\Models\Discount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class DiscountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Fetch discounts with pagination, 10 per page for a compact view
        $discounts = Discount::orderBy('name')->paginate(10);
        return view('discounts.index', compact('discounts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Define discount types for the select dropdown
        $discountTypes = ['percentage', 'fixed_amount']; 
        return view('discounts.create', compact('discountTypes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:percentage,fixed_amount', // Validate against allowed types
            'value' => 'required|numeric|min:0',
            'start_date' => 'required|date_format:Y-m-d\TH:i', // Matches HTML datetime-local format
            'end_date' => 'required|date_format:Y-m-d\TH:i|after_or_equal:start_date',
            'is_active' => 'boolean',
        ]);

        try {
            Discount::create($validated);
            return redirect()->route('discounts.index')->with('success', 'Discount added successfully!');
        } catch (\Exception $e) {
            Log::error('Add Discount Error: ' . $e->getMessage());
            return redirect()->back()->withInput()->withErrors(['error' => 'Failed to add discount: ' . $e->getMessage()]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Discount $discount)
    {
        $discountTypes = ['percentage', 'fixed_amount'];
        return view('discounts.edit', compact('discount', 'discountTypes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Discount $discount)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:percentage,fixed_amount',
            'value' => 'required|numeric|min:0',
            'start_date' => 'required|date_format:Y-m-d\TH:i',
            'end_date' => 'required|date_format:Y-m-d\TH:i|after_or_equal:start_date',
            'is_active' => 'boolean',
        ]);

        try {
            $discount->update($validated);
            return redirect()->route('discounts.index')->with('success', 'Discount updated successfully!');
        } catch (\Exception | \Illuminate\Database\QueryException $e) { // Catch QueryException for unique constraint etc.
            Log::error('Update Discount Error: ' . $e->getMessage());
            return redirect()->back()->withInput()->withErrors(['error' => 'Failed to update discount: ' . $e->getMessage()]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Discount $discount)
    {
        try {
            $discount->delete();
            return redirect()->route('discounts.index')->with('success', 'Discount deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Delete Discount Error: ' . $e->getMessage());
            return redirect()->back()->withErrors(['error' => 'Failed to delete discount: ' . $e->getMessage()]);
        }
    }
}