<?php

namespace App\Http\Controllers;

use App\Models\CentralSaleReport; // Use the new CentralSaleReport model
use App\Models\CentralSaleItem;   // Use the new CentralSaleItem model
use App\Models\CentralShowroom;  // Use the new CentralShowroom model
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class DmySalesController extends Controller
{
    /**
     * Display a listing of sales data (for web view) or return JSON data (for API).
     * This method now queries the central consolidated database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // Initialize base query for central sales reports
        $salesQuery = CentralSaleReport::query()
            ->with('saleItems', 'centralShowroom'); // Eager load sale items and showroom

        // Initialize base query for central sale items (for product-wise report)
        $saleItemsQuery = CentralSaleItem::query()
            ->with('centralSaleReport.centralShowroom'); // Eager load the parent report and its showroom

        // --- Date Filters ---
        $day = $request->input('day');
        $month = $request->input('month');
        $year = $request->input('year');
        $fromDate = $request->input('from_date');
        $toDate = $request->input('to_date');

        $periodDescription = "All Time";

        if ($day && $month && $year) {
            $salesQuery->whereDate('sale_date', Carbon::create($year, $month, $day)->toDateString());
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($day, $month, $year) {
                $q->whereDate('sale_date', Carbon::create($year, $month, $day)->toDateString());
            });
            $periodDescription = Carbon::create($year, $month, $day)->format('F j, Y');
        } elseif ($month && $year) {
            $salesQuery->whereMonth('sale_date', $month)->whereYear('sale_date', $year);
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($month, $year) {
                $q->whereMonth('sale_date', $month)->whereYear('sale_date', $year);
            });
            $periodDescription = Carbon::createFromDate($year, $month, 1)->format('F Y');
        } elseif ($year) {
            $salesQuery->whereYear('sale_date', $year);
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($year) {
                $q->whereYear('sale_date', $year);
            });
            $periodDescription = $year;
        } elseif ($fromDate && $toDate) {
            $salesQuery->whereBetween('sale_date', [Carbon::parse($fromDate)->startOfDay(), Carbon::parse($toDate)->endOfDay()]);
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($fromDate, $toDate) {
                $q->whereBetween('sale_date', [Carbon::parse($fromDate)->startOfDay(), Carbon::parse($toDate)->endOfDay()]);
            });
            $periodDescription = "from " . Carbon::parse($fromDate)->format('M j, Y') . " to " . Carbon::parse($toDate)->format('M j, Y');
        } elseif ($fromDate) {
            $salesQuery->where('sale_date', '>=', Carbon::parse($fromDate)->startOfDay());
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($fromDate) {
                $q->where('sale_date', '>=', Carbon::parse($fromDate)->startOfDay());
            });
            $periodDescription = "from " . Carbon::parse($fromDate)->format('M j, Y');
        } elseif ($toDate) {
            $salesQuery->where('sale_date', '<=', Carbon::parse($toDate)->endOfDay());
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($toDate) {
                $q->where('sale_date', '<=', Carbon::parse($toDate)->endOfDay());
            });
            $periodDescription = "up to " . Carbon::parse($toDate)->format('M j, Y');
        }


        // --- Other Filters (Product-related) ---
        $productName = $request->input('productname');
        $productCategory = $request->input('productcategory');
        $productionDepartment = $request->input('productiondepartment');
        $barcode = $request->input('barcode');
        $sku = $request->input('sku');
        $customNo = $request->input('custom_no');

        // Apply product filters to both sales and sale items queries (now on CentralSaleItem)
        if ($productName) {
            $salesQuery->whereHas('saleItems', function ($q) use ($productName) {
                $q->where('product_name', 'like', "%{$productName}%");
            });
            $saleItemsQuery->where('product_name', 'like', "%{$productName}%");
        }
        if ($productCategory) {
            $salesQuery->whereHas('saleItems', function ($q) use ($productCategory) {
                $q->where('product_category', 'like', "%{$productCategory}%");
            });
            $saleItemsQuery->where('product_category', 'like', "%{$productCategory}%");
        }
        if ($productionDepartment) {
            $salesQuery->whereHas('saleItems', function ($q) use ($productionDepartment) {
                $q->where('production_department', 'like', "%{$productionDepartment}%");
            });
            $saleItemsQuery->where('production_department', 'like', "%{$productionDepartment}%");
        }
        if ($barcode) {
            $salesQuery->whereHas('saleItems', function ($q) use ($barcode) {
                $q->where('barcode', 'like', "%{$barcode}%");
            });
            $saleItemsQuery->where('barcode', 'like', "%{$barcode}%");
        }
        if ($sku) {
            $salesQuery->whereHas('saleItems', function ($q) use ($sku) {
                $q->where('sku', 'like', "%{$sku}%");
            });
            $saleItemsQuery->where('sku', 'like', "%{$sku}%");
        }
        if ($customNo) {
            $salesQuery->whereHas('saleItems', function ($q) use ($customNo) {
                $q->where('custom_no', 'like', "%{$customNo}%");
            });
            $saleItemsQuery->where('custom_no', 'like', "%{$customNo}%");
        }

        // Showroom Filter (applied to both queries on showroom_name field or central_showroom_id)
        $showroomFilterValue = $request->input('showroom_filter'); // This can be 'all' or a central_showroom_id
        $showroomNameForDisplay = 'All Showrooms';

        if ($showroomFilterValue && $showroomFilterValue !== 'all') {
            // Filter by central_showroom_id
            $salesQuery->where('central_showroom_id', $showroomFilterValue);
            $saleItemsQuery->whereHas('centralSaleReport', function ($q) use ($showroomFilterValue) {
                $q->where('central_showroom_id', $showroomFilterValue);
            });
            // Get the showroom name for display
            $selectedShowroom = CentralShowroom::find($showroomFilterValue);
            if ($selectedShowroom) {
                $showroomNameForDisplay = $selectedShowroom->name;
            }
        }


        // --- Execute Queries ---
        $sales = $salesQuery->orderBy('sale_date', 'asc')->get();
        $saleItems = $saleItemsQuery->orderBy('created_at', 'asc')->get();

        // --- Report 1: Overall Sales Report ---
        $report1Data = [];
        $report1TotalSale = 0;
        $report1TotalDiscount = 0;
        $report1TotalNetSale = 0;

        $salesByDate = $sales->groupBy(function ($sale) {
            return Carbon::parse($sale->sale_date)->format('Y-m-d');
        });

        foreach ($salesByDate as $date => $dailySales) {
            $totalSale = $dailySales->sum('total_price');
            $totalDiscount = $dailySales->sum('discount_amount');
            $netSale = $totalSale - $totalDiscount;

            $report1Data[] = [
                'date' => Carbon::parse($date)->format('Y-m-d'),
                'total_sale' => number_format($totalSale, 2),
                'discount' => number_format($totalDiscount, 2),
                'net_sale' => number_format($netSale, 2),
            ];
            $report1TotalSale += $totalSale;
            $report1TotalDiscount += $totalDiscount;
            $report1TotalNetSale += $netSale;
        }

        // --- Report 2: Product-wise Sales Report ---
        $report2Data = [];
        $report2TotalAmount = 0;

        foreach ($saleItems as $item) {
            if ($item->centralSaleReport) { // Ensure parent report relationship exists
                $amount = ($item->quantity * $item->unit_price) - $item->discount_amount;
                $report2Data[] = [
                    'date' => Carbon::parse($item->centralSaleReport->sale_date)->format('Y-m-d'),
                    'productname' => $item->product_name,
                    'productcategory' => $item->product_category,
                    'productiondepartment' => $item->production_department,
                    'amount' => number_format($amount, 2),
                ];
                $report2TotalAmount += $amount;
            }
        }

        // --- Prepare Response ---
        $response = [
            'report1' => [
                'heading1' => "Sales Report of {$showroomNameForDisplay}",
                'heading2' => "For the period of {$periodDescription}",
                'columns' => ['Date', 'Total Sale', 'Discount', 'Net Sale'],
                'data' => $report1Data,
                'total' => [
                    'total_sale' => number_format($report1TotalSale, 2),
                    'total_discount' => number_format($report1TotalDiscount, 2),
                    'total_net_sale' => number_format($report1TotalNetSale, 2),
                ],
            ],
            'report2' => [
                'heading1' => "Product-wise Sales Report of {$showroomNameForDisplay}",
                'heading2' => "For the period of {$periodDescription}",
                'columns' => ['Date', 'Product Name', 'Product Category', 'Production Department', 'Amount'],
                'data' => $report2Data,
                'total' => [
                    'total_amount' => number_format($report2TotalAmount, 2),
                ],
            ],
            'filters_applied' => $request->all(),
        ];

        // If it's an API call or AJAX, return JSON
        if ($request->wantsJson() || $request->ajax()) {
            return response()->json($response);
        } else {
            // Otherwise, render the web view with initial filter data
            // Fetch distinct showroom names from CentralShowroom for the filter dropdown
            $centralShowrooms = CentralShowroom::orderBy('name')->get(); // Get all central showrooms
            $productCategories = CentralSaleItem::distinct('product_category')->pluck('product_category');
            $productionDepartments = CentralSaleItem::distinct('production_department')->pluck('production_department');
            $productNames = CentralSaleItem::distinct('product_name')->pluck('product_name');

            $fromDate = $request->input('from_date');
            $toDate = $request->input('to_date');
            $showroom_filter = $request->input('showroom_filter'); // Use the new filter name
            $productname = $request->input('productname');
            $productcategory = $request->input('productcategory');
            $productiondepartment = $request->input('productiondepartment');
            $barcode = $request->input('barcode');
            $sku = $request->input('sku');
            $custom_no = $request->input('custom_no');

            return view('pos.dmy_sales_report', compact(
                'centralShowrooms', 'productCategories', 'productionDepartments', 'productNames',
                'fromDate', 'toDate', 'showroom_filter', 'productname', 'productcategory',
                'productiondepartment', 'barcode', 'sku', 'custom_no'
            ));
        }
    }

    /**
     * Get distinct product categories for filter dropdown.
     */
    public function getProductCategories()
    {
        $categories = CentralSaleItem::distinct('product_category')->pluck('product_category');
        return response()->json($categories);
    }

    /**
     * Get distinct production departments for filter dropdown.
     */
    public function getProductionDepartments()
    {
        $departments = CentralSaleItem::distinct('production_department')->pluck('production_department');
        return response()->json($departments);
    }

    /**
     * Get distinct product names for filter dropdown.
     */
    public function getProductNames()
    {
        $productNames = CentralSaleItem::distinct('product_name')->pluck('product_name');
        return response()->json($productNames);
    }

    /**
     * Get distinct showrooms for filter dropdown (from central DB).
     */
    public function getShowrooms()
    {
        $showrooms = CentralShowroom::orderBy('name')->get(); // Fetch from CentralShowroom model
        return response()->json($showrooms);
    }
}
