<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\ProductPrice;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DrinksController extends Controller
{
    /**
     * Show the drinks report page.
     */
    public function index()
    {
        // Get unique years available in product stocks for filters
        $years = ProductStock::select(DB::raw('YEAR(stock_date) as year'))
                            ->distinct()
                            ->orderBy('year', 'desc')
                            ->pluck('year');

        $currentYear = Carbon::now()->year;
        if (!$years->contains($currentYear)) {
            $years->prepend($currentYear);
        }
        $years = $years->sortDesc()->values();

        // Determine the category string based on your explicit instruction.
        // If your database actually uses separate categories 'Soft Drinks' AND 'Juices',
        // you MUST change this to: $categoryFilter = ['Soft Drinks', 'Juices'];
        // and then use whereIn('productcategory', $categoryFilter) below.
        $categoryFilter = 'Soft Drinks & Juices'; // Assuming this is the exact string in your productcategory column

        Log::info('DrinksController@index: Product Category Filter used: ' . $categoryFilter);

        // Fetch products by their productcategory, selecting only necessary fields from Product
        $drinkProducts = Product::where('productcategory', $categoryFilter)
            ->select('id', 'productname', 'productcategory')
            ->with(['prices' => function($query) {
                // IMPORTANT: Fetching price ONLY from the ProductPrice model.
                // For initial header rendering, we use the current date to get the latest effective price.
                $todayFormatted = Carbon::now()->format('Y-m-d');
                Log::info('DrinksController@index: Price query for current date: ' . $todayFormatted);

                $query->where('effective_start_date', '<=', $todayFormatted)
                      ->where(function($q) use ($todayFormatted) {
                          $q->whereNull('effective_end_date')
                            ->orWhere('effective_end_date', '>=', $todayFormatted);
                      })
                      ->orderBy('effective_start_date', 'desc')
                      ->orderBy('id', 'desc')
                      ->take(1);
            }])
            ->get();

        // Prepare products data for the frontend headers
        $productHeaders = $drinkProducts->map(function($product) {
            // Retrieve the price from the loaded ProductPrice relationship
            $price = $product->prices->first() ? $product->prices->first()->price : 'N/A';
            Log::info("DrinksController@index: Product ID: {$product->id}, Product Name: {$product->productname}, Determined Price: {$price}");

            return [
                'id' => $product->id,
                'name' => $product->productname, // From Product model
                'category' => $product->productcategory, // From Product model
                'price' => $price, // ONLY from ProductPrice model
            ];
        })->sortBy('name')->values()->toArray();

        

        return view('reports.drinksreport', compact('years', 'productHeaders'));
    }

    /**
     * Fetch report data for Soft Drinks & Juices based on filters.
     */
    public function getReportData(Request $request)
    {
        $month = $request->input('month');
        $year = $request->input('year');

        if (empty($year) || empty($month)) {
            Log::info('DrinksController@getReportData: Empty year or month received. Returning empty data.');
            return response()->json([
                'reportData' => [],
                'productHeaders' => [],
                'productTotals' => [],
                'grandTotal' => 0
            ], 200);
        }

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfMonth();
        $endDate = Carbon::createFromDate($year, $month, 1)->endOfMonth();

       

        $categoryFilter = 'Soft Drinks & Juices';
        

        $drinkProducts = Product::where('productcategory', $categoryFilter)
            ->select('id', 'productname', 'productcategory')
            ->orderBy('productname')
            ->with(['prices' => function($query) use ($startDate) {
                $startDateFormatted = $startDate->format('Y-m-d');
                $query->where('effective_start_date', '<=', $startDateFormatted)
                      ->where(function($q) use ($startDateFormatted) {
                          $q->whereNull('effective_end_date')
                            ->orWhere('effective_end_date', '>=', $startDateFormatted);
                      })
                      ->orderBy('effective_start_date', 'desc')
                      ->orderBy('id', 'desc')
                      ->take(1);
            }])
            ->get();

        $productIds = $drinkProducts->pluck('id')->toArray();
        

        // --- START STOCK DATA DEBUGGING ---
        DB::connection()->enableQueryLog(); // Enable query log
        $stockDataRaw = ProductStock::select(
                DB::raw('DATE_FORMAT(stock_date, "%Y-%m-%d") as report_date'),
                'product_id',
                DB::raw('SUM(closing_stock) as total_closing_stock_qty')
            )
            ->whereIn('product_id', $productIds)
            ->whereBetween('stock_date', [$startDate->format('Y-m-d'), $endDate->format('Y-m-d')])
            ->groupBy('report_date', 'product_id')
            ->get();
        $queries = DB::getQueryLog();
        DB::connection()->disableQueryLog(); // Disable query log

        

        // Group the raw results by date
        $stockData = $stockDataRaw->groupBy('report_date');
        
        // --- END STOCK DATA DEBUGGING ---

        $reportData = [];
        $productTotals = array_fill_keys($productIds, 0);
        $grandTotal = 0;

        for ($date = $startDate->copy(); $date->lte($endDate); $date->addDay()) {
            $currentDate = $date->format('Y-m-d');
            $dailyRow = [
                'date' => $currentDate,
                'items' => [],
                'dailyTotal' => 0,
            ];

            $dailyStock = $stockData->get($currentDate);

            foreach ($drinkProducts as $product) {
                $closingStockQty = 0;
                if ($dailyStock) {
                    $stockEntry = $dailyStock->firstWhere('product_id', $product->id);
                    if ($stockEntry) {
                        $closingStockQty = (int)$stockEntry->total_closing_stock_qty;
                    }
                }
                $dailyRow['items'][$product->id] = $closingStockQty;
                $dailyRow['dailyTotal'] += $closingStockQty;
                $productTotals[$product->id] += $closingStockQty;
            }
            $grandTotal += $dailyRow['dailyTotal'];
            $reportData[] = $dailyRow;
        }

        $productHeaders = $drinkProducts->map(function($product) use ($startDate) {
            $price = $product->prices()
                             ->where('effective_start_date', '<=', $startDate->format('Y-m-d'))
                             ->where(function($q) use ($startDate) {
                                 $q->whereNull('effective_end_date')
                                   ->orWhere('effective_end_date', '>=', $startDate->format('Y-m-d'));
                             })
                             ->orderBy('effective_start_date', 'desc')
                             ->orderBy('id', 'desc')
                             ->value('price');

            return [
                'id' => $product->id,
                'name' => $product->productname,
                'category' => $product->productcategory,
                'price' => $price ?? 'N/A',
            ];
        })->sortBy('name')->values()->toArray();

        

        return response()->json([
            'reportData' => $reportData,
            'productHeaders' => $productHeaders,
            'productTotals' => $productTotals,
            'grandTotal' => $grandTotal
        ]);
    }
}