<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Apishowroom; // Ensure this is the correct model for API-enabled showrooms
use Illuminate\Support\Facades\Log; // For logging errors or debug information
use App\Models\ProductPrice;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException; // Import for validation exceptions

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::query();

        if ($request->search) {
            // Original search on productname, extended to new fields
            $query->where(function($q) use ($request) {
                $q->where('productname', 'LIKE', '%' . $request->search . '%')
                  ->orWhere('barcode', 'LIKE', '%' . $request->search . '%')       // New field search
                  ->orWhere('sku', 'LIKE', '%' . $request->search . '%')           // New field search
                  ->orWhere('custom_no', 'LIKE', '%' . $request->search . '%');    // New field search
            });
        }

        $products = $query->orderBy('id', 'desc')->paginate(40);
        // Fetch all available API-enabled showrooms
        $apishowrooms = Apishowroom::orderBy('name')->get();

        return view('fac.products.index', compact('products', 'apishowrooms'));
    }

    public function create()
    {
        // Fetch all available API-enabled showrooms to pass to the create view
        $apishowrooms = Apishowroom::orderBy('name')->get();
        return view('fac.products.create', compact('apishowrooms'));
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'dateofentry' => 'required|date',
                'productname' => 'required|string|max:255',
                'productiondepartment' => 'required|string|max:100',
                'productcategory' => 'required|string|max:100',
                'price' => 'required|numeric|min:0',
                'barcode' => 'nullable|string|max:255|unique:products,barcode', // New field validation
                'sku' => 'nullable|string|max:255|unique:products,sku',         // New field validation
                'custom_no' => 'nullable|string|max:255',                      // New field validation
                'unit' => 'nullable|string|max:50',                            // New field validation
                'status' => 'nullable|string|max:50',                          // New field validation
                'showroom_id' => 'nullable|exists:apishowrooms,id',            // New field validation (ensures showroom exists)
            ]);

            Product::create($validated);
            return redirect()->route('products.index')->with('success', 'Product created!');

        } catch (ValidationException $e) {
            // Catch validation errors and redirect back with input and errors
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            // Catch any other general exceptions
            Log::error("Error creating product: " . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'Failed to create product. Please try again.')->withInput();
        }
    }

    public function edit(Product $product)
    {
        // Fetch all available API-enabled showrooms to pass to the edit view
        $apishowrooms = Apishowroom::orderBy('name')->get();
        return view('fac.products.edit', compact('product', 'apishowrooms'));
    }

    public function update(Request $request, Product $product)
    {
        try {
            $validated = $request->validate([
                'dateofentry' => 'required|date',
                'productname' => 'required|string|max:255',
                'productiondepartment' => 'required|string|max:100',
                'productcategory' => 'required|string|max:100',
                'price' => 'required|numeric|min:0',
                'barcode' => 'nullable|string|max:255|unique:products,barcode,' . $product->id, // New field validation, ignoring current product
                'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,         // New field validation, ignoring current product
                'custom_no' => 'nullable|string|max:255',                      // New field validation
                'unit' => 'nullable|string|max:50',                            // New field validation
                'status' => 'nullable|string|max:50',                          // New field validation
                'showroom_id' => 'nullable|exists:apishowrooms,id',            // New field validation
            ]);

            $product->update($validated);
            return redirect()->route('products.index')->with('success', 'Product updated!');

        } catch (ValidationException $e) {
            // Catch validation errors and redirect back with input and errors
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            // Catch any other general exceptions
            Log::error("Error updating product ID {$product->id}: " . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'Failed to update product. Please try again.')->withInput();
        }
    }

    public function destroy(Product $product)
    {
        try {
            $product->delete();
            return redirect()->route('products.index')->with('success', 'Product deleted!');
        } catch (\Exception $e) {
            Log::error("Error deleting product ID {$product->id}: " . $e->getMessage());
            // Check for foreign key constraint violation
            if (str_contains($e->getMessage(), 'Integrity constraint violation')) {
                return redirect()->back()->with('error', 'Failed to delete product. It is linked to existing stock or price records.');
            }
            return redirect()->back()->with('error', 'Failed to delete product. Please try again.');
        }
    }

    public function syncInternalProductToProductPrices(Request $request)
    {
        // Ensure only authenticated users can perform this action
        if (!Auth::check()) {
            Log::warning('Attempted unauthorized syncInternalProductToProductPrices access.');
            return response()->json(['message' => 'Unauthorized action.'], 403);
        }

        $syncDate = Carbon::createFromFormat('Y-m-d', '2025-06-01')->startOfDay(); // Define the sync date

        // Fetch all products
        $products = Product::all();

        $syncedCount = 0;
        $errors = [];

        foreach ($products as $product) {
            try {
                // Find or create a ProductPrice entry for this product and effective_start_date
                // We're looking for a general price, not a showroom-specific override,
                // so we include is_showroom_override = false and showroom_id = NULL in the search criteria.
                ProductPrice::updateOrCreate(
                    [
                        'product_id' => $product->id,
                        'effective_start_date' => $syncDate,
                        'is_showroom_override' => false, // Important: Assume general price sync
                        'showroom_id' => null,           // Important: Assume general price sync
                    ],
                    [
                        'price' => $product->price,
                        'changed_by_user_id' => Auth::id(), // ID of the logged-in user
                        'reason' => 'Initial sync from Products table or price update',
                        'effective_end_date' => null, // Or set to a far future date if you prefer
                    ]
                );
                $syncedCount++;
            } catch (\Exception $e) {
                // Log or collect errors for products that failed to sync
                $errors[] = "Failed to sync product ID {$product->id} ({$product->productname}): " . $e->getMessage();
                Log::error("ProductPrice sync error: " . $e->getMessage(), ['product_id' => $product->id]);
            }
        }

        if (!empty($errors)) {
            return response()->json([
                'message' => "Synchronization completed with {$syncedCount} successes and " . count($errors) . " errors.",
                'errors' => $errors
            ], 200); // Or 500 if you consider any error a failure
        }

        return response()->json(['message' => "Successfully synced {$syncedCount} product prices."]);
    }
}