<?php

namespace App\Http\Controllers;

use App\Models\Product; // 
use App\Models\ProductPrice;
use App\Models\User; // 
use App\Models\Showroom; 
use App\Models\Apishowroom; 
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log; // For logging errors or debug information

class ProductPriceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
        {
            // Fetch all product prices with eager loaded relationships for efficiency
            $query = ProductPrice::with(['product', 'changedByUser', 'showroom']);

            // --- Existing Filter Logic ---
            // ... (your filter conditions like product_name, price_min, dates, etc.) ...
            // Example (keep all your existing filters here):
            if ($request->filled('product_name')) {
                $query->whereHas('product', function ($q) use ($request) {
                    $q->where('productname', 'like', '%' . $request->input('product_name') . '%');
                });
            }
            // ... (all other filters) ...

            if ($request->filled('showroom_id')) {
                $query->where('showroom_id', $request->input('showroom_id'));
            }
            // --- End Existing Filter Logic ---

            // Order and paginate results
            $productPrices = $query->latest('effective_start_date')->paginate(50)->withQueryString();

            // Fetch auxiliary data for filter dropdowns
            $products = Product::all(['id', 'productname']);
            $users = User::all(['id', 'name']);

            // Fetch general Showroom data for the filter dropdown
            // This is assuming your 'product_prices.showroom_id' column links to your main 'showrooms' table.
            $showrooms = Showroom::all(['id', 'name']);

            // Fetch API Showroom data for the "Push Prices" modal dropdown
            // This is critical for the '$apishowrooms' variable.
            $apishowrooms = Apishowroom::where('is_active', true)
                                    ->whereNotNull('api_url')
                                    ->get(['id', 'name', 'api_url', 'api_token']);

            // --- DEBUGGING STEP: Temporarily add this line to see what $apishowrooms contains ---
            // dd($apishowrooms);
            // If this shows an empty collection or an error, the issue is with the query for Apishowroom.
            // If it shows data, the issue is in the Blade file.

            return view('product_prices.index', compact('productPrices', 'products', 'users', 'showrooms', 'apishowrooms')); // <--- Ensure 'apishowrooms' is in compact()
        }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $products = Product::all(['id', 'productname']); // Get product list for select2
        $users = User::all(['id', 'name']); // Get users list for select2
        $showrooms = Showroom::all(['id', 'name']); // Get showrooms list for select2

        return view('product_prices.create', compact('products', 'users', 'showrooms'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'price' => ['required', 'numeric', 'min:0'], // Use numeric for decimal validation
            'effective_start_date' => ['required', 'date'],
            'effective_end_date' => ['nullable', 'date', 'after_or_equal:effective_start_date'],
            'changed_by_user_id' => ['nullable', 'exists:users,id'],
            'reason' => ['nullable', 'string', 'max:255'],
            'is_showroom_override' => ['boolean'],
            'showroom_id' => ['nullable', 'exists:showrooms,id', 'required_if:is_showroom_override,true'],
        ]);

        // Logic to handle effective_end_date for existing prices:
        // If a new price is being set for a product, and effective_end_date is null,
        // then the *previous* active price for that product needs its effective_end_date set.
        if (empty($validatedData['effective_end_date'])) {
            $productPrice = ProductPrice::where('product_id', $validatedData['product_id'])
                                        ->whereNull('effective_end_date')
                                        ->first();
            if ($productPrice) {
                // Ensure the new price doesn't start before the old one ends unless it's explicitly set
                // For simplicity, let's just set the old one to end when the new one begins.
                // In a real-world scenario, you might want more complex overlap validation.
                $productPrice->update(['effective_end_date' => $validatedData['effective_start_date']]);
            }
        }


        ProductPrice::create($validatedData);

        return redirect()->route('product_prices.index')->with('success', 'Product price created successfully!');
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ProductPrice $productPrice)
    {
        $products = Product::all(['id', 'productname']);
        $users = User::all(['id', 'name']);
        $showrooms = Showroom::all(['id', 'name']);

        return view('product_prices.edit', compact('productPrice', 'products', 'users', 'showrooms'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ProductPrice $productPrice)
    {
        $validatedData = $request->validate([
            'product_id' => ['required', 'exists:products,id'], // Product ID should not change after creation for a price record
            'price' => ['required', 'numeric', 'min:0'],
            'effective_start_date' => ['required', 'date'],
            'effective_end_date' => ['nullable', 'date', 'after_or_equal:effective_start_date'],
            'changed_by_user_id' => ['nullable', 'exists:users,id'],
            'reason' => ['nullable', 'string', 'max:255'],
            'is_showroom_override' => ['boolean'],
            'showroom_id' => ['nullable', 'exists:showrooms,id', 'required_if:is_showroom_override,true'],
        ]);

        // When updating, we generally update the specific price record.
        // If the effective_end_date is being set to null, and another null effective_end_date exists for the same product,
        // you might want to handle it. For now, this update simply modifies the current record.
        // Complex logic for managing overlapping/current prices is better handled on price *creation*
        // or via a dedicated "activate new price" action.

        $productPrice->update($validatedData);

        return redirect()->route('product_prices.index')->with('success', 'Product price updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProductPrice $productPrice)
    {
        $productPrice->delete();
        return redirect()->route('product_prices.index')->with('success', 'Product price deleted successfully!');
    }



}