<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\Permission; // Assuming you have a Permission model for the relationship
use Illuminate\Http\Request;

class RoleController extends Controller
{
    public function index()
    {
        $roles = Role::paginate(10); // Adjust pagination as needed
        return view('roles.index', compact('roles'));
    }

    public function create()
    {
        $permissions = Permission::all(); // Fetch all permissions for the select box
        return view('roles.create', compact('permissions'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'guard_name' => 'required|string|max:255',
            'permissions' => 'nullable|array', // For attaching permissions
        ]);

        $role = Role::create($request->only(['name', 'guard_name']));
        if ($request->has('permissions')) {
            $role->permissions()->sync($request->input('permissions'));
        }

        return redirect()->route('roles.index')->with('success', 'Role created successfully.');
    }

    public function edit(Role $role)
    {
        $permissions = Permission::all(); // Fetch all permissions for the select box
        $assignedPermissions = $role->permissions->pluck('id')->toArray(); // Get IDs of permissions already assigned
        return view('roles.edit', compact('role', 'permissions', 'assignedPermissions'));
    }

    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:roles,name,' . $role->id,
            'guard_name' => 'required|string|max:255',
            'permissions' => 'nullable|array', // For attaching permissions
        ]);

        $role->update($request->only(['name', 'guard_name']));
        if ($request->has('permissions')) {
            $role->permissions()->sync($request->input('permissions'));
        } else {
            $role->permissions()->detach(); // Detach all if no permissions selected
        }

        return redirect()->route('roles.index')->with('success', 'Role updated successfully.');
    }

    public function destroy(Role $role)
    {
        $role->delete();
        return redirect()->route('roles.index')->with('success', 'Role deleted successfully.');
    }
}