<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role; // Assuming this is your custom Role model, if not, remove. SpatieRole is also used.
use App\Models\Showroom; // <<< ADDED: Import the Showroom model
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role as SpatieRole; // Using Spatie's Role model

class UserController extends Controller
{
    // public function __construct()
    // {
    //     // Restrict access to admins using Spatie's role middleware
    //     // Re-enable this if you want to apply a global middleware for this controller
    //     // $this->middleware(['auth', 'role:admin']);
    // }

    public function index(Request $request)
    {
        $search = $request->input('search');
        $users = User::query()
            ->when($search, function ($query, $search) {
                return $query->where('name', 'like', "%{$search}%")
                             ->orWhere('email', 'like', "%{$search}%");
            })
            ->with('roles', 'showroom') // <<< MODIFIED: Load roles AND showroom for display
            ->paginate(20);

        return view('users.index', compact('users', 'search'));
    }

    public function create()
    {
        $roles = SpatieRole::all()->pluck('name', 'id');
        $showrooms = Showroom::all(); // <<< ADDED: Fetch all showrooms
        return view('users.create', compact('roles', 'showrooms')); // <<< MODIFIED: Pass showrooms to the view
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email',
            'password' => 'required|string|max:255',
            'roles' => 'required|exists:roles,id',
            'showroom_id' => 'nullable|exists:showrooms,id', // <<< ADDED: Validate showroom_id
            'email_verified_at' => 'nullable|date',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'showroom_id' => $validated['showroom_id'] ?? null, // <<< ADDED: Save showroom_id
            'email_verified_at' => $validated['email_verified_at'] ?? null, // Handle nullable email_verified_at
        ]);

        $user->roles()->sync([$validated['roles']]);

        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }

    public function edit(User $user)
    {
        $roles = SpatieRole::all()->pluck('name', 'id');
        $showrooms = Showroom::all(); // <<< ADDED: Fetch all showrooms
        return view('users.edit', compact('user', 'roles', 'showrooms')); // <<< MODIFIED: Pass showrooms to the view
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|max:255',
            'roles' => 'required|exists:roles,id',
            'showroom_id' => 'nullable|exists:showrooms,id', // <<< ADDED: Validate showroom_id
            'email_verified_at' => 'nullable|date',
        ]);

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => $validated['password'] ? Hash::make($validated['password']) : $user->password,
            'showroom_id' => $validated['showroom_id'] ?? null, // <<< ADDED: Update showroom_id
            'email_verified_at' => $validated['email_verified_at'] ?? null, // Handle nullable email_verified_at
        ]);

        $user->roles()->sync([$validated['roles']]);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }
}