<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

/**
 * Class Product
 * 
 * @property int $id
 * @property Carbon $dateofentry
 * @property string|null $productname
 * @property string|null $productiondepartment
 * @property string|null $productcategory
 * @property float|null $price
 *
 * @package App\Models
 */
class Product extends Model
{
	protected $table = 'products';
	public $timestamps = false;

	protected $casts = [
		'dateofentry' => 'datetime',
		'price' => 'float',
		'showroom_id' => 'int'
	];

	protected $fillable = [
		'dateofentry',
		'productname',
		'productiondepartment',
		'productcategory',
		'price',
		'barcode',
		'sku',
		'custom_no',
		'unit',
		'status',
		'showroom_id'
	];

	public function stocks()
{
    return $this->hasMany(ProductStock::class);
}

public function prices()
{
    return $this->hasMany(ProductPrice::class, 'product_id');
}

public function showroom()
    {
        return $this->belongsTo(Showroom::class, 'showroom_id');
    }

    /**
     * Determine if the product is sold by weight.
     */
    public function getIsWeighedItemAttribute(): bool
    {
        // Assuming 'kg' or 'gram' units indicate a weighed item.
        // It's highly recommended to have a dedicated 'is_weighed_item' boolean column in the products table.
        return in_array(strtolower($this->unit), ['kg', 'kilogram', 'gram', 'g']);
    }

    /**
     * Get the current effective price for the product, considering showroom override.
     * Prioritizes ProductPrice entries, then falls back to Product.price.
     *
     * @param int|null $showroom_id The ID of the current showroom for override check.
     * @return float
     */
    public function getCurrentPrice(?int $showroom_id = null): float
    {
        // Base query for active prices in product_prices table
        $priceQuery = $this->prices()
                           ->where('effective_start_date', '<=', now())
                           ->where(function ($q) {
                               $q->whereNull('effective_end_date')
                                 ->orWhere('effective_end_date', '>=', now());
                           });

        // 1. Check for showroom-specific override price
        if ($showroom_id) {
            $showroomOverridePrice = (clone $priceQuery)
                                     ->where('is_showroom_override', true)
                                     ->where('showroom_id', $showroom_id)
                                     ->latest('effective_start_date') // Get the most recent effective price for this showroom
                                     ->first();
            if ($showroomOverridePrice) {
                return (float) $showroomOverridePrice->price;
            }
        }

        // 2. Fallback to global price from product_prices table
        $globalPrice = (clone $priceQuery)
                       ->where('is_showroom_override', false)
                       ->whereNull('showroom_id') // Ensure it's a global price
                       ->latest('effective_start_date') // Get the most recent effective global price
                       ->first();
        if ($globalPrice) {
            return (float) $globalPrice->price;
        }

        // 3. Fallback to the price directly on the product itself
        return (float) $this->price;
    }



}
