<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Product; // Your Product model
use App\Models\ProductPrice; // Your new ProductPrice model
use App\Models\User; // Assuming you have a User model
use Illuminate\Support\Facades\DB; // If you need to check for user existence

class InitialProductPricesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // 1. Define the default user ID.
        //    This user will be set as 'changed_by_user_id'.
        //    Ensure a user with this ID exists in your 'users' table,
        //    or create a system user if none exists.
        $defaultUserId = 1; // Adjust this if your default user ID is different.

        // Optional: Create a default system user if ID 1 doesn't exist
        if (!User::where('id', $defaultUserId)->exists()) {
             // You might want to log this or throw an error in a real scenario,
             // or create the user if it's truly a system user.
             // For a quick fix, let's create a placeholder if it's missing.
             User::create([
                 'id' => $defaultUserId,
                 'name' => 'System Migration',
                 'email' => 'system@example.com', // Use a dedicated email
                 'password' => bcrypt(uniqid()), // Random password
                 'created_at' => now(),
                 'updated_at' => now(),
             ]);
             $this->command->info("Created a default user for migration (ID: {$defaultUserId}).");
        }


        // 2. Define the common effective start date
        //    As requested: 01-06-2025 (June 1st, 2025)
        $effectiveStartDate = '2025-06-01 00:00:00';

        $this->command->info('Starting initial product price migration...');

        // 3. Fetch all products
        //    At this point, the 'products' table should still have the 'price' column.
        $products = Product::all();

        foreach ($products as $product) {
            // Check if the 'price' attribute exists and is not null
            // This handles cases where old products might not have a price set
            if (isset($product->price) && $product->price !== null) {
                // Ensure no duplicate price entries for the same product at the exact start date
                $existingPriceRecord = ProductPrice::where('product_id', $product->id)
                                                   ->where('effective_start_date', $effectiveStartDate)
                                                   ->first();

                if (!$existingPriceRecord) {
                    ProductPrice::create([
                        'product_id' => $product->id,
                        'price' => $product->price, // Get price from the old 'products' table
                        'effective_start_date' => $effectiveStartDate,
                        'effective_end_date' => null, // No end date, meaning it's the currently active price
                        'changed_by_user_id' => $defaultUserId,
                        'reason' => 'Initial price migration',
                        'is_showroom_override' => false, // These are central base prices
                        'showroom_id' => null, // No showroom involved in central migration
                    ]);
                    $this->command->info("  - Migrated price for product: {$product->productname} (ID: {$product->id})");
                } else {
                    $this->command->warn("  - Price record for product {$product->productname} (ID: {$product->id}) at {$effectiveStartDate} already exists. Skipping.");
                }
            } else {
                $this->command->warn("  - Product {$product->productname} (ID: {$product->id}) does not have a 'price' value. Skipping.");
            }
        }

        $this->command->info('Initial product price migration complete.');
    }
}