@extends('layouts.main')

@section('content')
<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2 class="h4 mb-0 text-dark">Generate API Token for Showroom</h2>
        <a href="{{ route('admin.tokens.index') }}" class="btn btn-secondary btn-sm">Manage Tokens</a> {{-- If you create a management view --}}
    </div>

    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <form id="generateTokenForm">
                @csrf 

                <div class="mb-3">
                    <label for="showroom_id" class="form-label">Select Showroom</label>
                    <select class="form-select" id="showroom_id" name="showroom_id" required>
                        <option value="">-- Select a Showroom --</option>
                        @foreach ($showrooms as $showroom)
                            <option value="{{ $showroom->id }}">{{ $showroom->name }}</option>
                        @endforeach
                    </select>
                    <div class="invalid-feedback" id="showroom_id_error"></div>
                </div>

                <div class="mb-3">
                    <label for="token_name" class="form-label">Token Name</label>
                    <input type="text" class="form-control" id="token_name" name="token_name" placeholder="e.g., Showroom X Daily Stock API" required>
                    <div class="invalid-feedback" id="token_name_error"></div>
                </div>

                <div class="mb-3">
                    <label for="abilities" class="form-label">Token Abilities</label>
                    <div>
                        @foreach ($availableAbilities as $key => $description)
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="ability_{{ $key }}" name="abilities[]" value="{{ $key }}" checked>
                                <label class="form-check-label" for="ability_{{ $key }}">{{ $description }}</label>
                            </div>
                        @endforeach
                        <div class="form-check form-check-inline">
                            <input class="form-check-input" type="checkbox" id="ability_all" name="abilities[]" value="*" checked>
                            <label class="form-check-label" for="ability_all">All (Full Access - Use with Caution)</label>
                        </div>
                    </div>
                    <div class="invalid-feedback" id="abilities_error"></div>
                    <small class="form-text text-muted">Select what actions this token is authorized to perform.</small>
                </div>

                <div class="mb-3">
                    <label for="expires_at" class="form-label">Expires At (Optional)</label>
                    <input type="date" class="form-control" id="expires_at" name="expires_at" min="{{ now()->format('Y-m-d') }}">
                    <div class="invalid-feedback" id="expires_at_error"></div>
                    <small class="form-text text-muted">Leave empty for a token that never expires. Set a date for temporary tokens.</small>
                </div>

                <button type="submit" class="btn btn-primary">Generate Token</button>
            </form>
        </div>
    </div>

    <div class="card shadow-sm" id="tokenResultCard" style="display:none;">
        <div class="card-header">Generated API Token</div>
        <div class="card-body">
            <div class="alert alert-success" role="alert">
                <p><strong>Showroom:</strong> <span id="resultShowroomName"></span></p>
                <p><strong>Token Name:</strong> <span id="resultTokenName"></span></p>
                <p><strong>Expires At:</strong> <span id="resultExpiresAt"></span></p>
                <p class="mb-0"><strong>Your New API Token:</strong></p>
                <div class="input-group">
                    <input type="text" class="form-control" id="generatedToken" readonly>
                    <button class="btn btn-outline-secondary" type="button" id="copyTokenBtn">Copy</button>
                </div>
                <small class="form-text text-danger">
                    Please copy this token NOW. It will not be shown again for security reasons.
                    This token is sensitive and should be stored securely by the showroom's system.
                </small>
            </div>
        </div>
    </div>

    <div class="alert alert-danger mt-3" role="alert" id="errorAlert" style="display:none;">
        <strong>Error:</strong> <span id="errorMessage"></span>
        <ul id="errorList"></ul>
    </div>

</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const form = document.getElementById('generateTokenForm');
        const tokenResultCard = document.getElementById('tokenResultCard');
        const generatedTokenInput = document.getElementById('generatedToken');
        const copyTokenBtn = document.getElementById('copyTokenBtn');
        const resultShowroomName = document.getElementById('resultShowroomName');
        const resultTokenName = document.getElementById('resultTokenName');
        const resultExpiresAt = document.getElementById('resultExpiresAt');
        const errorAlert = document.getElementById('errorAlert');
        const errorMessageSpan = document.getElementById('errorMessage');
        const errorList = document.getElementById('errorList');

        // Function to clear validation feedback
        function clearValidationFeedback() {
            document.querySelectorAll('.is-invalid').forEach(el => {
                el.classList.remove('is-invalid');
            });
            document.querySelectorAll('.invalid-feedback').forEach(el => {
                el.textContent = '';
            });
            errorAlert.style.display = 'none';
            errorList.innerHTML = '';
        }

        form.addEventListener('submit', async function(e) {
            e.preventDefault();
            clearValidationFeedback(); // Clear previous errors

            const formData = new FormData(form);
            const data = Object.fromEntries(formData.entries());

            // Handle abilities (checkboxes)
            const abilities = Array.from(form.querySelectorAll('input[name="abilities[]"]:checked'))
                                 .map(cb => cb.value);
            data.abilities = abilities;

            try {
                const response = await fetch('{{ route('admin.tokens.generate-showroom-token') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                        // IMPORTANT: For an admin panel, the admin user needs an API token
                        // or this route should be protected by web middleware if it's a traditional SPA.
                        // Assuming the admin user is logged in via web sessions for this form.
                        // If admin also uses Sanctum API tokens for their panel:
                        // 'Authorization': 'Bearer YOUR_ADMIN_API_TOKEN'
                    },
                    body: JSON.stringify(data)
                });

                const result = await response.json();

                if (response.ok) {
                    tokenResultCard.style.display = 'block';
                    generatedTokenInput.value = result.token;
                    resultShowroomName.textContent = result.showroom_name;
                    resultTokenName.textContent = result.token_name;
                    resultExpiresAt.textContent = result.expires_at ? result.expires_at : 'Never';

                    // Scroll to the result
                    tokenResultCard.scrollIntoView({ behavior: 'smooth' });

                    // Optionally, clear the form after successful generation
                    form.reset();
                } else {
                    errorAlert.style.display = 'block';
                    errorMessageSpan.textContent = result.message || 'An unexpected error occurred.';
                    if (result.errors) {
                        for (const field in result.errors) {
                            const input = document.getElementById(field);
                            if (input) {
                                input.classList.add('is-invalid');
                                const feedbackDiv = document.getElementById(`${field}_error`);
                                if (feedbackDiv) {
                                    feedbackDiv.textContent = result.errors[field][0];
                                }
                            } else {
                                // Add to general error list if no specific input for field
                                const li = document.createElement('li');
                                li.textContent = `${field}: ${result.errors[field][0]}`;
                                errorList.appendChild(li);
                            }
                        }
                    }
                }

            } catch (error) {
                console.error('Network or other error:', error);
                errorAlert.style.display = 'block';
                errorMessageSpan.textContent = 'Could not connect to the server. Please check your network.';
            }
        });

        copyTokenBtn.addEventListener('click', function() {
            generatedTokenInput.select();
            generatedTokenInput.setSelectionRange(0, 99999); // For mobile devices
            document.execCommand('copy');
            alert('Token copied to clipboard!');
        });
    });
</script>
@endsection