@extends('layouts.app') // Assuming a layout for central office

@section('content')
<div class="container-fluid">
    <h2 class="mb-4">Central Stock Reports</h2>

    <div class="card mb-4">
        <div class="card-header">Filters</div>
        <div class="card-body">
            <form id="reportFilters" class="row g-3 align-items-end">
                <div class="col-md-3">
                    <label for="dateFrom" class="form-label">Date From</label>
                    <input type="date" class="form-control" id="dateFrom" value="{{ date('Y-m-01') }}">
                </div>
                <div class="col-md-3">
                    <label for="dateTo" class="form-label">Date To</label>
                    <input type="date" class="form-control" id="dateTo" value="{{ date('Y-m-d') }}">
                </div>
                <div class="col-md-4">
                    <label for="showroomSelect" class="form-label">Select Showrooms</label>
                    <select id="showroomSelect" class="form-select" multiple>
                        </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">Apply Filters</button>
                </div>
            </form>
        </div>
    </div>

    <div class="card mb-4">
        <div class="card-header">Aggregated Summary</div>
        <div class="card-body">
            <table class="table table-sm table-bordered" id="centralSummaryTable">
                <thead>
                    <tr>
                        <th>Showroom</th>
                        <th>OS Value</th>
                        <th>Received Value</th>
                        <th>R/D Value</th>
                        <th>CS Value</th>
                        <th>Sale Value</th>
                    </tr>
                </thead>
                <tbody>
                    <tr><td colspan="6" class="text-center">Select filters and apply to view summary.</td></tr>
                </tbody>
                <tfoot id="grandTotalSummary"></tfoot>
            </table>
        </div>
    </div>

    <div class="card">
        <div class="card-header">Detailed Stock Report</div>
        <div class="card-body">
            <table class="table table-sm table-bordered" id="centralDetailedTable">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Showroom</th>
                        <th>Product</th>
                        <th>Category</th>
                        <th>OS</th>
                        <th>Received</th>
                        <th>R/D</th>
                        <th>CS</th>
                        <th>Sale Qty</th>
                        <th>Unit Price</th>
                        <th>Sale Value</th>
                    </tr>
                </thead>
                <tbody>
                    <tr><td colspan="11" class="text-center">Select filters and apply to view detailed report.</td></tr>
                </tbody>
            </table>
            <div id="paginationLinks" class="d-flex justify-content-center mt-3"></div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const dateFromInput = document.getElementById('dateFrom');
        const dateToInput = document.getElementById('dateTo');
        const showroomSelect = document.getElementById('showroomSelect');
        const reportFiltersForm = document.getElementById('reportFilters');
        const centralSummaryTableBody = document.querySelector('#centralSummaryTable tbody');
        const grandTotalSummaryFoot = document.getElementById('grandTotalSummary');
        const centralDetailedTableBody = document.querySelector('#centralDetailedTable tbody');
        const paginationLinks = document.getElementById('paginationLinks');

        let currentPage = 1;

        // Fetch showrooms for the filter dropdown
        function fetchShowrooms() {
            fetch('/api/central/showrooms', {
                headers: {
                    'Accept': 'application/json',
                    'Authorization': 'Bearer YOUR_CENTRAL_OFFICE_API_TOKEN' // Replace with actual token
                }
            })
            .then(response => response.json())
            .then(showrooms => {
                showroomSelect.innerHTML = '';
                showrooms.forEach(showroom => {
                    const option = document.createElement('option');
                    option.value = showroom.id;
                    option.textContent = showroom.name;
                    showroomSelect.appendChild(option);
                });
                // Initialize Bootstrap Select (if using a library for multi-select)
                // e.g., $(showroomSelect).selectpicker();
            })
            .catch(error => {
                console.error('Error fetching showrooms:', error);
                alert('Could not load showrooms.');
            });
        }

        // Fetch and render summary report
        function fetchSummaryReport(filters) {
            const params = new URLSearchParams(filters);
            fetch(`/api/central/stock-summary?${params.toString()}`, {
                headers: {
                    'Accept': 'application/json',
                    'Authorization': 'Bearer YOUR_CENTRAL_OFFICE_API_TOKEN'
                }
            })
            .then(response => response.json())
            .then(data => {
                centralSummaryTableBody.innerHTML = '';
                grandTotalSummaryFoot.innerHTML = ''; // Clear previous grand totals

                let grandOS = 0, grandReceived = 0, grandRD = 0, grandCS = 0, grandSale = 0;

                if (data.length === 0) {
                    centralSummaryTableBody.innerHTML = '<tr><td colspan="6" class="text-center">No summary data found for the selected filters.</td></tr>';
                } else {
                    data.forEach(item => {
                        const row = `
                            <tr>
                                <td>${item.showroom_name}</td>
                                <td class="text-end">${item.total_opening_value.toFixed(2)}</td>
                                <td class="text-end">${item.total_received_value.toFixed(2)}</td>
                                <td class="text-end">${item.total_return_damage_value.toFixed(2)}</td>
                                <td class="text-end">${item.total_closing_value.toFixed(2)}</td>
                                <td class="text-end fw-bold">${item.total_sale_value.toFixed(2)}</td>
                            </tr>
                        `;
                        centralSummaryTableBody.insertAdjacentHTML('beforeend', row);

                        grandOS += item.total_opening_value;
                        grandReceived += item.total_received_value;
                        grandRD += item.total_return_damage_value;
                        grandCS += item.total_closing_value;
                        grandSale += item.total_sale_value;
                    });

                    // Add grand total row
                    grandTotalSummaryFoot.innerHTML = `
                        <tr>
                            <th>Grand Total</th>
                            <th class="text-end">${grandOS.toFixed(2)}</th>
                            <th class="text-end">${grandReceived.toFixed(2)}</th>
                            <th class="text-end">${grandRD.toFixed(2)}</th>
                            <th class="text-end">${grandCS.toFixed(2)}</th>
                            <th class="text-end fw-bold">${grandSale.toFixed(2)}</th>
                        </tr>
                    `;
                }
            })
            .catch(error => {
                console.error('Error fetching summary report:', error);
                centralSummaryTableBody.innerHTML = '<tr><td colspan="6" class="text-danger text-center">Failed to load summary.</td></tr>';
            });
        }

        // Fetch and render detailed report
        function fetchDetailedReport(filters, page = 1) {
            filters.page = page;
            const params = new URLSearchParams(filters);
            fetch(`/api/central/stock-report?${params.toString()}`, {
                headers: {
                    'Accept': 'application/json',
                    'Authorization': 'Bearer YOUR_CENTRAL_OFFICE_API_TOKEN'
                }
            })
            .then(response => response.json())
            .then(data => {
                centralDetailedTableBody.innerHTML = '';
                paginationLinks.innerHTML = '';

                if (data.data.length === 0) {
                    centralDetailedTableBody.innerHTML = '<tr><td colspan="11" class="text-center">No detailed stock data found for the selected filters.</td></tr>';
                } else {
                    data.data.forEach(item => {
                        const row = `
                            <tr>
                                <td>${item.stock_date}</td>
                                <td>${item.showroom.name}</td>
                                <td>${item.product.name}</td>
                                <td>${item.product.category}</td>
                                <td class="text-end">${item.opening_stock}</td>
                                <td class="text-end">${item.received}</td>
                                <td class="text-end">${item.return_damage}</td>
                                <td class="text-end">${item.closing_stock}</td>
                                <td class="text-end">${item.sale_quantity}</td>
                                <td class="text-end">${item.product.price.toFixed(2)}</td>
                                <td class="text-end fw-bold">${item.sale_value.toFixed(2)}</td>
                            </tr>
                        `;
                        centralDetailedTableBody.insertAdjacentHTML('beforeend', row);
                    });

                    // Render pagination links
                    if (data.last_page > 1) {
                        const ul = document.createElement('ul');
                        ul.className = 'pagination';
                        data.links.forEach(link => {
                            const li = document.createElement('li');
                            li.className = `page-item ${link.active ? 'active' : ''} ${link.url ? '' : 'disabled'}`;
                            const a = document.createElement('a');
                            a.className = 'page-link';
                            a.href = '#'; // Prevent actual navigation
                            a.innerHTML = link.label
                                .replace(/&laquo; Previous/, '&laquo;')
                                .replace(/Next &raquo;/, '&raquo;');
                            if (link.url) {
                                a.addEventListener('click', (e) => {
                                    e.preventDefault();
                                    const url = new URL(link.url);
                                    const newPage = url.searchParams.get('page');
                                    fetchDetailedReport(filters, newPage);
                                });
                            }
                            li.appendChild(a);
                            ul.appendChild(li);
                        });
                        paginationLinks.appendChild(ul);
                    }
                }
            })
            .catch(error => {
                console.error('Error fetching detailed report:', error);
                centralDetailedTableBody.innerHTML = '<tr><td colspan="11" class="text-danger text-center">Failed to load detailed report.</td></tr>';
            });
        }

        // Handle form submission
        reportFiltersForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const filters = {
                date_from: dateFromInput.value,
                date_to: dateToInput.value,
                showroom_ids: Array.from(showroomSelect.selectedOptions).map(option => option.value)
            };
            fetchSummaryReport(filters);
            fetchDetailedReport(filters);
        });

        // Initial load
        fetchShowrooms();
        reportFiltersForm.dispatchEvent(new Event('submit')); // Trigger initial report load
    });
</script>
@endsection