@extends('layouts.main')

@section('content')
<div class="container-fluid small">
    <div class="d-flex justify-content-between mb-3 align-items-center">
        <h6 class="fw-bold">Daily Stock Entry</h6>
        <input type="date" id="stockDate" class="form-control form-control-sm w-auto" value="{{ date('Y-m-d') }}">
    </div>

    <div id="stockTable" class="border rounded mb-3"></div>

    <div class="text-end">
        <button class="btn btn-sm btn-primary" id="submitBtn">Save Stock</button>
    </div>
</div>
@endsection


@section('scripts')
<script>
let jsp, flatData = [], categoryMap = {};

document.addEventListener('DOMContentLoaded', () => {
    const dateInput = document.getElementById('stockDate');
    loadData(dateInput.value);
    dateInput.addEventListener('change', e => loadData(e.target.value));
    document.getElementById('submitBtn').addEventListener('click', saveData);
});

function loadData(date) {
    fetch(`{{ route('stocks.data') }}?date=${date}`)
        .then(res => res.json())
        .then(({ data }) => {
            flatData = [];
            categoryMap = {};
            let rowIdx = 0;

            Object.entries(data).forEach(([category, products]) => {
                // Insert category row
                flatData.push([
                    `▼ ${category.toUpperCase()}`, '', '', '', '', '', '', 'group'
                ]);
                const groupRow = rowIdx++;
                categoryMap[groupRow] = [];

                products.forEach(prod => {
                    flatData.push([
                        prod.productname,
                        prod.price,
                        prod.opening_stock,
                        prod.received,
                        prod.return_damage,
                        prod.closing_stock,
                        prod.sale,
                        prod.product_id
                    ]);
                    categoryMap[groupRow].push(rowIdx++);
                });
            });

            if (jsp) jsp.destroy();

            jsp = jspreadsheet(document.getElementById('stockTable'), {
                data: flatData.map(r => r.slice(0, 7)),
                columns: [
                    { title: 'Product', width: 200, readOnly: true },
                    { title: 'Price', type: 'numeric', readOnly: true },
                    { title: 'OS', type: 'numeric' },
                    { title: 'Received', type: 'numeric' },
                    { title: 'R/D', type: 'numeric' },
                    { title: 'CS', type: 'numeric' },
                    { title: 'Sale', type: 'numeric', readOnly: true }
                ],
                rows: Object.fromEntries(
                    Object.keys(categoryMap).map(idx => [idx, { cssClass: 'bg-light fw-bold' }])
                ),
                onchange: (instance, cell, x, y, value) => {
                    const row = jsp.getRowData(y);
                    if (flatData[y][7] !== 'group') {
                        const os = +row[2] || 0;
                        const rec = +row[3] || 0;
                        const rd = +row[4] || 0;
                        const cs = +row[5] || 0;
                        const sale = os + rec - rd - cs;
                        jsp.setValueFromCoords(6, y, sale, true);
                    }
                },
                onclick: (el, x, y) => {
                    if (flatData[y][7] === 'group' && x === 0) {
                        const hidden = jsp.getRowVisibility(categoryMap[y][0]) === false;
                        categoryMap[y].forEach(row => jsp.setRowVisibility(row, !hidden));
                        const label = flatData[y][0];
                        flatData[y][0] = (hidden ? '▼' : '►') + label.slice(1);
                        jsp.setValueFromCoords(0, y, flatData[y][0], true);
                    }
                },
                freezeColumns: 1
            });
        });
}

function saveData() {
    const date = document.getElementById('stockDate').value;
    const payload = {};

    flatData.forEach((row, i) => {
        const prodID = row[7];
        if (prodID !== 'group' && prodID != null) {
            const r = jsp.getRowData(i);
            payload[prodID] = {
                opening_stock: +r[2] || 0,
                received: +r[3] || 0,
                return_damage: +r[4] || 0,
                closing_stock: +r[5] || 0
            };
        }
    });

    fetch("{{ route('stocks.update') }}", {
        method: "POST",
        headers: {
            'X-CSRF-TOKEN': "{{ csrf_token() }}",
            'Content-Type': "application/json"
        },
        body: JSON.stringify({
            stock_date: date,
            entries: payload
        })
    })
    .then(res => res.json())
    .then(data => alert(data.message))
    .catch(() => alert("Failed to save stock"));
}
</script>

<style>
.bg-light.fw-bold {
    background: #eef6ff !important;
    font-weight: bold;
    color: #444;
    text-align: center;
}
</style>
@endsection
